<?php
// Start the session to manage login state.
session_start();

// If an admin is already logged in, they shouldn't see the login page again.
// Redirect them directly to the admin dashboard.
if (isset($_SESSION['admin_id'])) {
    header("location: index.php");
    exit;
}

// Include the main database connection file.
// The path is '../' because this file is in the 'admin' folder, one level deep.
require_once '../includes/db_connect.php';

// Initialize a variable to hold any error messages.
$message = '';

// Process the form only if it has been submitted via the POST method.
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    
    // Get and sanitize the inputs.
    $mobile = trim($_POST['mobile']);
    $password = trim($_POST['password']);

    // Use a prepared statement to securely fetch the user's data.
    // Crucially, it checks for `role = 'admin'` to ensure only admins can log in here.
    $stmt = $conn->prepare("SELECT id, full_name, password, status FROM users WHERE mobile = ? AND role = 'admin'");
    $stmt->bind_param("s", $mobile);
    $stmt->execute();
    $result = $stmt->get_result();

    // Check if exactly one admin account was found.
    if ($result->num_rows == 1) {
        $admin = $result->fetch_assoc();
        
        // First, check if the admin account is active.
        if ($admin['status'] === 'Inactive') {
            $message = '<div class="alert alert-danger">আপনার এডমিন একাউন্টটি নিষ্ক্রিয় করা হয়েছে।</div>';
        }
        // If active, verify the password.
        elseif (password_verify($password, $admin['password'])) {
            // Password is correct. Login successful.
            
            // Set session variables to remember the admin is logged in.
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_name'] = $admin['full_name'];
            
            // Redirect to the main admin dashboard.
            header("location: index.php");
            exit; // Stop script execution after redirect.
        } else {
            // If password does not match.
            $message = '<div class="alert alert-danger">আপনার দেওয়া পাসওয়ার্ডটি সঠিক নয়।</div>';
        }
    } else {
        // If no admin account was found with the given mobile number.
        $message = '<div class="alert alert-danger">এই মোবাইল নম্বর দিয়ে কোনো এডমিন একাউন্ট খুঁজে পাওয়া যায়নি।</div>';
    }
    $stmt->close();
}
$conn->close();
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>এডমিন লগইন</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body { background-color: #f8f9fa; font-family: 'Noto Sans Bengali', sans-serif; }
        .login-container { 
            max-width: 400px; 
            margin: 150px auto; 
            background: #fff; 
            padding: 30px; 
            border-radius: 10px; 
            box-shadow: 0 4px 8px rgba(0,0,0,0.1);
        }
    </style>
</head>
<body>
    <div class="login-container text-center">
        <h2>এডমিন প্যানেল</h2>
        <p class="text-muted">লগইন করে ادامه দিন</p>

        <?php if(!empty($message)) echo $message; ?>
        
        <form method="post" class="mt-4">
            <div class="form-floating mb-3">
                <input type="tel" name="mobile" class="form-control" id="floatingInput" placeholder="Mobile Number" required>
                <label for="floatingInput">মোবাইল নম্বর</label>
            </div>
            <div class="form-floating">
                <input type="password" name="password" class="form-control" id="floatingPassword" placeholder="Password" required>
                <label for="floatingPassword">পাসওয়ার্ড</label>
            </div>
            <button type="submit" class="btn btn-primary w-100 py-2 mt-3">লগইন</button>
        </form>
    </div>
</body>
</html>