<?php
// Include the database connection file, which also starts the session.
require_once 'includes/db_connect.php';

// Security Check: If the user is not logged in, redirect them to the login page.
if (!isset($_SESSION["user_id"])) {
    header("location: login.php");
    exit;
}

// Get the logged-in user's ID from the session.
$user_id = $_SESSION['user_id'];

// Fetch all transactions for this specific user, ordered by the most recent first.
$transactions = [];
$stmt = $conn->prepare("SELECT type, amount, description, status, created_at FROM transactions WHERE user_id = ? ORDER BY created_at DESC");
$stmt->bind_param("i", $user_id);
$stmt->execute();
$result = $stmt->get_result();

// If transactions are found, store them in an array.
if ($result->num_rows > 0) {
    while ($row = $result->fetch_assoc()) {
        $transactions[] = $row;
    }
}
$stmt->close();
$conn->close();

/**
 * Helper function to translate English transaction types from the database into Bengali for display.
 * @param string $type The transaction type in English (e.g., 'Package Purchase').
 * @return string The translated Bengali string.
 */
function translate_type_to_bengali($type) {
    $translations = [
        'Recharge'         => 'রিচার্জ',
        'Balance Transfer' => 'ব্যালেন্স ট্রান্সফার',
        'Balance Received' => 'ব্যালেন্স গ্রহণ',
        'Add Balance'      => 'ব্যালেন্স যোগ',
        'Package Purchase' => 'প্যাকেজ ক্রয়',
        'Bill Payment'     => 'বিল পেমেন্ট',
        'Admin Deposit'    => 'এডমিন ডিপোজিট',
        'Admin Withdraw'   => 'এডমিন উইথড্র'
    ];
    // Return the translated string, or the original if no translation is found.
    return $translations[$type] ?? htmlspecialchars($type);
}
?>
<!DOCTYPE html>
<html lang="bn">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>লেনদেনের বিবরণী</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        body { background-color: #f0f2f5; font-family: 'Noto Sans Bengali', sans-serif; }
        .statement-container { max-width: 700px; margin: 30px auto; padding: 20px; background: #fff; box-shadow: 0 0 15px rgba(0,0,0,0.1); border-radius: 10px; }
        .page-header { display: flex; align-items: center; margin-bottom: 20px; }
        .page-header a { font-size: 20px; color: #333; text-decoration: none; margin-right: 15px; }
        .page-header h2 { font-size: 22px; margin: 0; }
        .transaction-item { display: flex; justify-content: space-between; align-items: center; padding: 15px 10px; border-bottom: 1px solid #eee; }
        .transaction-item:last-child { border-bottom: none; }
        .transaction-details { display: flex; align-items: center; }
        .transaction-icon { font-size: 20px; width: 45px; height: 45px; border-radius: 50%; display: flex; align-items: center; justify-content: center; margin-right: 15px; }
        .icon-sent { background-color: #ffebee; color: #e53935; } /* Red for sent/debit */
        .icon-received { background-color: #e8f5e9; color: #43a047; } /* Green for received/credit */
        .transaction-info p { margin: 0; font-size: 14px; color: #666; }
        .transaction-info strong { color: #333; font-size: 16px; }
        .transaction-amount { text-align: right; }
        .amount-sent { color: #e53935; font-weight: bold; }
        .amount-received { color: #43a047; font-weight: bold; }
        .transaction-amount span { font-size: 12px; color: #888; display: block; }
    </style>
</head>
<body>
    <div class="container">
        <div class="statement-container">
            <div class="page-header">
                <a href="index.php"><i class="fas fa-arrow-left"></i></a>
                <h2>স্টেটমেন্ট</h2>
            </div>
            
            <div class="transaction-list">
                <?php if (empty($transactions)): ?>
                    <p class="text-center text-muted mt-4">এখনো কোনো লেনদেন হয়নি।</p>
                <?php else: ?>
                    <?php foreach ($transactions as $txn): ?>
                        <?php
                            // Determine if the transaction is a debit (money sent) or credit (money received)
                            $isSent = in_array($txn['type'], ['Recharge', 'Balance Transfer', 'Package Purchase', 'Bill Payment', 'Admin Withdraw']);
                            $iconClass = $isSent ? 'fa-arrow-up icon-sent' : 'fa-arrow-down icon-received';
                            $amountClass = $isSent ? 'amount-sent' : 'amount-received';
                            $amountPrefix = $isSent ? '-' : '+';
                        ?>
                        <div class="transaction-item">
                            <div class="transaction-details">
                                <div class="transaction-icon <?php echo explode(' ', $iconClass)[1]; ?>">
                                    <i class="fas <?php echo explode(' ', $iconClass)[0]; ?>"></i>
                                </div>
                                <div class="transaction-info">
                                    <strong><?php echo translate_type_to_bengali($txn['type']); ?></strong>
                                    <p><?php echo htmlspecialchars($txn['description']); ?></p>
                                </div>
                            </div>
                            <div class="transaction-amount">
                                <strong class="<?php echo $amountClass; ?>"><?php echo $amountPrefix; ?> ৳<?php echo number_format($txn['amount'], 2); ?></strong>
                                <span><?php echo date("d M, Y h:i A", strtotime($txn['created_at'])); ?></span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
</body>
</html>