<?php

/**
 * =========================================================================
 * MOCK API SIMULATION FILE
 * =========================================================================
 * This file contains functions that simulate responses from external third-party APIs
 * like utility companies or SMS gateways. In a real-world, production application,
 * you would replace the logic inside these functions with actual calls to the
 * real API providers (e.g., using cURL).
 * =========================================================================
 */

/**
 * Simulates fetching bill details from a utility company's API.
 * This function mimics checking a bill's status and amount.
 *
 * @param string $biller        The name of the biller (e.g., 'DESCO Postpaid').
 * @param string $accountNumber The customer's account/consumer number.
 * @return array                An associative array with bill details or an error message.
 */
function fetchBillDetailsAPI($biller, $accountNumber) {
    // --- Simulation Logic ---
    // We return different pre-defined responses based on the account number provided for testing.

    // Simulate a successful response for a specific, known test account number.
    if ($accountNumber === '123456789') {
        return [
            'status' => 'success',
            'customerName' => 'Sample Test Customer',
            'amount' => 1250.75,
            'dueDate' => '2025-10-25', // Example due date
            'biller' => $biller
        ];
    }
    // Simulate a response for a bill that has already been paid.
    elseif ($accountNumber === '987654321') {
        return [
            'status' => 'error',
            'message' => 'এই বিলটি ইতোমধ্যে পরিশোধ করা হয়েছে।'
        ];
    }
    // Simulate a "bill not found" or "invalid account number" response for all other numbers.
    else {
        return [
            'status' => 'error',
            'message' => 'আপনার দেওয়া একাউন্ট নম্বরের বিপরীতে কোনো বকেয়া বিল পাওয়া যায়নি।'
        ];
    }
}

/**
 * Simulates processing a bill payment through a third-party API.
 * This function mimics the final payment confirmation step.
 *
 * @param string $biller        The name of the biller.
 * @param string $accountNumber The customer's account number.
 * @param float  $amount        The amount that is being paid.
 * @return array                A success or failure response from the simulated API.
 */
function processBillPaymentAPI($biller, $accountNumber, $amount) {
    // For our simulation, we will always assume the payment is successful.
    return [
        'status' => 'success',
        // Generate a unique, fake transaction ID for the receipt.
        'transactionId' => 'BP' . time() . rand(100, 999), 
        'message' => 'বিল পেমেন্ট সফলভাবে সম্পন্ন হয়েছে।'
    ];
}

/**
 * Simulates sending an OTP to a user's mobile number via an SMS gateway.
 * @param string $mobile The recipient's mobile number.
 * @param string $otp The 6-digit OTP code.
 * @return bool Always returns true for simulation purposes.
 */
function sendOtpAPI($mobile, $otp) {
    // In a real application, you would have code here to call your SMS provider's API.
    // Example: $response = file_get_contents("https://sms-provider.com/api?to=$mobile&text=Your OTP is $otp");
    
    // For testing purposes, we will store the OTP in a session variable.
    // This allows us to see the "sent" OTP on the verification page without receiving a real SMS.
    // IMPORTANT: Remove this line in a live production environment with a real SMS API.
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
    $_SESSION['debug_otp'] = $otp;
    
    // We always return true to simulate a successful SMS delivery.
    return true;
}

?>